<?php
/**
 * Test Email Sending Script
 * Simple interface to test the email sending functionality
 */

require_once 'send_email.php';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $testEmail = $_POST['email'] ?? '';
    
    if (empty($testEmail)) {
        $response = [
            'status' => 'error',
            'message' => 'Please provide an email address'
        ];
    } elseif (!filter_var($testEmail, FILTER_VALIDATE_EMAIL)) {
        $response = [
            'status' => 'error',
            'message' => 'Please provide a valid email address'
        ];
    } else {
        $response = sendTestEmail($testEmail);
    }
    
    // Return JSON response for AJAX requests
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
        strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        echo json_encode($response);
        exit;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SMTP Email Test</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            background: white;
            border-radius: 10px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            max-width: 500px;
            width: 100%;
            padding: 40px;
        }
        
        h1 {
            color: #333;
            margin-bottom: 10px;
            font-size: 28px;
        }
        
        .subtitle {
            color: #666;
            margin-bottom: 30px;
            font-size: 14px;
        }
        
        .config-info {
            background: #f5f5f5;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 25px;
            font-size: 12px;
        }
        
        .config-info h3 {
            color: #333;
            margin-bottom: 10px;
            font-size: 14px;
        }
        
        .config-info p {
            color: #666;
            margin: 5px 0;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            color: #333;
            margin-bottom: 8px;
            font-weight: 500;
        }
        
        input[type="email"] {
            width: 100%;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 5px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        input[type="email"]:focus {
            outline: none;
            border-color: #667eea;
        }
        
        button {
            width: 100%;
            padding: 12px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 5px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s, box-shadow 0.2s;
        }
        
        button:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(102, 126, 234, 0.4);
        }
        
        button:active {
            transform: translateY(0);
        }
        
        button:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .alert {
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            display: none;
        }
        
        .alert.show {
            display: block;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .spinner {
            display: none;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255,255,255,0.3);
            border-top-color: white;
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
            margin: 0 auto;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>📧 SMTP Email Test</h1>
        <p class="subtitle">Test your SMTP configuration by sending a test email</p>
        
        <div class="config-info">
            <h3>Current Configuration:</h3>
            <p><strong>Host:</strong> smtpout.secureserver.net</p>
            <p><strong>Port:</strong> 465 (SSL)</p>
            <p><strong>From:</strong> info@dextratechnologies.com</p>
        </div>
        
        <div id="alert" class="alert"></div>
        
        <form id="emailForm" method="POST">
            <div class="form-group">
                <label for="email">Recipient Email Address:</label>
                <input 
                    type="email" 
                    id="email" 
                    name="email" 
                    placeholder="Enter email address to test"
                    required
                >
            </div>
            
            <button type="submit" id="submitBtn">
                <span id="btnText">Send Test Email</span>
                <div class="spinner" id="spinner"></div>
            </button>
        </form>
    </div>
    
    <script>
        const form = document.getElementById('emailForm');
        const alert = document.getElementById('alert');
        const submitBtn = document.getElementById('submitBtn');
        const btnText = document.getElementById('btnText');
        const spinner = document.getElementById('spinner');
        
        form.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            // Disable button and show spinner
            submitBtn.disabled = true;
            btnText.style.display = 'none';
            spinner.style.display = 'block';
            alert.classList.remove('show');
            
            const formData = new FormData(form);
            
            try {
                const response = await fetch('test.php', {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest'
                    }
                });
                
                const result = await response.json();
                
                // Show alert
                alert.className = 'alert show alert-' + result.status;
                alert.textContent = result.message;
                
                // Reset form on success
                if (result.status === 'success') {
                    form.reset();
                }
                
            } catch (error) {
                alert.className = 'alert show alert-error';
                alert.textContent = 'An error occurred. Please try again.';
            } finally {
                // Re-enable button and hide spinner
                submitBtn.disabled = false;
                btnText.style.display = 'block';
                spinner.style.display = 'none';
            }
        });
    </script>
</body>
</html>
